import fs from 'fs';
import { Logger } from 'koishi';
import { execFile } from 'child_process';
import { promisify } from 'util';
import path from 'path';
import { getPackageJsonPackages } from './utils/read-package-json';
import ConfigLoader from '@koishijs/loader';
import { KoishiConfig, PluginInstallConfig } from './def/interfaces';
const execFileAsync = promisify(execFile);

const logger = new Logger('bootstrap-install-plugin');

async function checkPluginExists(
  name: string,
  allowCommunity: boolean,
  allowOfficial: boolean,
) {
  const deps = await getPackageJsonPackages();
  const entry = Object.entries(deps).find(
    ([packageName, version]) =>
      packageName === name ||
      (allowOfficial && packageName === `@koishijs/plugin-${name}`) ||
      (allowCommunity && packageName.endsWith(`koishi-plugin-${name}`)),
  );
  if (!entry) {
    return;
  }
  try {
    const stat = await fs.promises.stat(path.join('node_modules', entry[0]));
    if (!stat?.isDirectory()) {
      return;
    }
    return entry[1];
  } catch (e) {
    return;
  }
}

async function npmInstall(name: string) {
  logger.info(`Installing package ${name}.`);
  try {
    await execFileAsync('npm', [
      'install',
      '--save-exact',
      '--loglevel',
      'error',
      name,
    ]);
    logger.info(`Package ${name} installed.`);
    return true;
  } catch (e) {
    logger.warn(`Package ${name} not found.`);
    return false;
  }
}
async function tryInstallPackages(names: string[]) {
  if (process.env.DRY_RUN) {
    logger.info(`Will install packages ${names.join(', ')}.`);
    return true;
  }
  for (const name of names) {
    if (await npmInstall(name)) {
      return true;
    }
  }
  return false;
}

async function installPlugin(name: string, info: PluginInstallConfig) {
  const suffixMatching = name.match(/(.+):.+/);
  if (suffixMatching) {
    name = suffixMatching[1];
  }

  if (name.match(/^([\.\/~\\]|[A-Za-z]:[\/\\])/)) {
    logger.info(`Plugin ${name} is a local plugin, skipping.`);
    return;
  }

  const version =
    info.$version ||
    (typeof info.$install === 'string' ? info.$install : undefined);

  logger.info(`Installing plugin ${name}@${version || '*'}.`);
  const allowCommunity = !info.$official;
  const allowOfficial = !info.$community;
  if (!allowCommunity && !allowOfficial) {
    logger.warn(`Plugin ${name} is neither official nor community, skipping.`);
    return;
  }
  const existingPluginVersion = await checkPluginExists(
    name,
    allowCommunity,
    allowOfficial,
  );
  if (
    existingPluginVersion &&
    (existingPluginVersion === version || !version || version === '*')
  ) {
    logger.info(`Plugin ${name}@${existingPluginVersion} exists, skipping.`);
    return;
  }
  const installList: string[] = [];
  const communityPrefix = 'koishi-plugin-';
  const officialPrefix = '@koishijs/plugin-';
  const installVersion = version || 'latest';
  if (name.includes(communityPrefix) || name.startsWith(officialPrefix)) {
    installList.push(`${name}@${installVersion}`);
  } else if (name.startsWith('@') && name.includes('/')) {
    const [author, pluginName] = name.split('/');
    const installName = pluginName.startsWith(communityPrefix)
      ? pluginName
      : `${communityPrefix}${pluginName}`;
    installList.push(
      `${author}/${communityPrefix}${installName}@${installVersion}`,
    );
  } else {
    if (allowOfficial) {
      installList.push(`${officialPrefix}${name}@${installVersion}`);
    }
    if (allowCommunity) {
      installList.push(`${communityPrefix}${name}@${installVersion}`);
    }
  }
  if (!installList.length) {
    logger.warn(`Plugin ${name} has nothing to install.`);
    return;
  }
  const result = await tryInstallPackages(installList);
  if (!result) {
    logger.error(`Plugin ${name}@${version || 'unknown'} install failed.`);
  }
}

export async function installPluginEntry() {
  logger.info(`Bootstrapping...`);
  const config = new ConfigLoader<KoishiConfig>().readConfig();
  const plugins = config?.plugins;
  if (!plugins) {
    logger.warn(`No plugins found, exiting.`);
    return;
  }
  // console.log(config.plugins);
  logger.info(`Cleaning NPM cache.`);
  await execFileAsync('npm', ['cache', 'clean', '--force']);
  let queue = Object.entries(plugins).reverse();
  while (queue.length) {
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    const [name, info] = queue.pop()!;
    if (name.startsWith('~') || !info) {
      // disabled plugin
      continue;
    }
    if (name.startsWith('group:')) {
      queue = queue.concat(Object.entries(info).reverse());
      continue;
    }
    if (!info.$install && !info.$version) {
      continue;
    }
    await installPlugin(name, info);
  }
  logger.info(`Bootstrap finished.`);
}
