import { fromLong } from 'ip';
import _ from 'lodash';
import { createHash } from 'crypto';

function gcd(a: number, b: number) {
  if (b === 0) {
    return a;
  }
  return gcd(b, a % b);
}

function gcds(n: number[]) {
  return n.reduce((a, b) => gcd(a, b));
}

function formatIp(value: number, mask: number) {
  return `${fromLong(value)}/${fromLong(mask)}`;
}

export interface Rule {
  //srcValue: number;
  //dstValue: number;
  src: string;
  dst: string;
}

function compactWeights(weights: number[]) {
  const gcd = gcds(weights);
  return weights.map((w) => w / gcd);
}

function rulesOfMaskLength(length: number) {
  const srcMask = 0x1;
  const dstMask = (0x1 << length) - 1;
  const ruleCount = Math.pow(2, length + 1);
  const rules: Rule[] = [];
  for (let i = 0; i < ruleCount; i++) {
    const srcValue = i & 0x1;
    const dstValue = (i >> 1) & dstMask;
    rules.push({
      //srcValue,
      //dstValue,
      src: formatIp(srcValue, srcMask),
      dst: formatIp(dstValue, dstMask),
    });
  }
  return rules;
}

interface Slot {
  rules: Rule[];
  total: number;
  remaining: number;
}

function selectSlot(rule: Rule, slots: Slot[]) {
  return _.minBy(
    slots.filter((s) =>
      slots.every(
        (other) =>
          s.rules.filter((r) => r.dst === rule.dst).length <=
          other.rules.filter((r) => r.dst === rule.dst).length,
      ),
    ),
    (s) => s.rules.filter((r) => r.src === rule.src).length,
  );
}

export function getRules(_weights: number[]) {
  const weights = compactWeights(_weights);
  const maskLength = Math.ceil(Math.log2(_.sum(weights)));
  const rules = rulesOfMaskLength(maskLength);
  const slots: Slot[] = weights.map((v) => ({
    rules: [] as Rule[],
    total: v,
    remaining: v,
  }));
  let availableSlots = [...slots];
  for (const rule of rules) {
    const currentSlot = selectSlot(rule, availableSlots);
    currentSlot.rules.push(rule);
    --currentSlot.remaining;
    if (currentSlot.remaining === 0) {
      availableSlots = availableSlots.filter((s) => s !== currentSlot);
      if (!availableSlots.length) {
        availableSlots = [...slots];
        availableSlots.forEach((s) => (s.remaining = s.total));
      }
    }
  }
  return slots.map((s) => s.rules);
}
