import 'reflect-metadata';
import { Context, GetEvents } from 'cordis';
import { ClassType, SchemaClass } from 'schemastery-gen';
import { MetadataSetter, Reflector } from 'typed-reflector';
import type {
  Awaitable,
  Condition,
  ControlType,
  PartialDeep,
  TypedMethodDecorator,
} from './def';
import { RegistrarAspect } from './registrar-aspect';
import Schema from 'schemastery';
import { PluginRegistrar } from './plugin-def';
import _ from 'lodash';
import { RegisterMeta } from './utility/register-meta';

declare module 'cordis' {
  interface Context {
    __parent?: any;
  }
}

// eslint-disable-next-line @typescript-eslint/no-namespace
export namespace Registrar {
  export interface MetadataArrayMap {
    CordisRegisterKeys: string;
    CordisContextTransformer: RegisterMeta<ContextTransformer<Context>>;
    CordisTopLevelAction: RegisterMeta<TopLevelActionDef<Context>>;
    CordisContextLayers: RegisterMeta<ContextCallbackLayer<Context>>;
    CordisControl: ControlType;
    CordisPluginUsing: string;
    CordisPluginProvide: ProvideDefinition<Context>;
    CordisPluginInjectKeys: string;
    CordisPluginSystemKeys: string;
  }
  export interface MetadataMap {
    CordisRegister: MethodMeta<Context>;
    CordisPluginInject: string;
    CordisPluginSystem: PluginRegistrar.SystemInjectFun<Context>;
    CordisPluginPredefineSchema: Schema | ClassType<any>;
    CordisPluginPredefineName: string;
    CordisPluginFork: PluginRegistrar.PluginClass<Context>;
    CordisPluginReusable: boolean;
  }

  export type DecorateFunctionParam<
    Ctx extends Context,
    A extends any[] = any[],
    R = any,
  > = (ctx: Ctx, ...args: A) => R;

  export type DecorateFunctionParamSingle<
    Ctx extends Context,
    P,
    A extends any[] = any[],
    R = any,
  > = (ctx: Ctx, param: P, ...args: A) => R;

  export type MethodResolver<
    Ctx extends Context,
    A extends any[] = any[],
    F extends (...args: any[]) => any = (...args: any[]) => any,
  > = DecorateFunctionParamSingle<Ctx, F, A>;

  export type MethodMeta<Ctx extends Context> = RegisterMeta<
    MethodResolver<Ctx>,
    { type: string }
  >;

  export type RegisterResult<Ctx extends Context, T> = Omit<
    MethodMeta<Ctx>,
    'run'
  > & {
    type: string;
    key: keyof T & string;
    result: any;
    ctx: Ctx;
  };

  export interface ProvideOptions extends Context.ServiceOptions {
    immediate?: boolean;
  }
  export interface ProvideDefinition<C extends Context> extends ProvideOptions {
    serviceName: string;
  }

  export type ContextFunction<Ctx extends Context, T> = DecorateFunctionParam<
    Ctx,
    [],
    T
  >;

  export type ContextTransformer<
    Ctx extends Context,
    A extends any[] = any[],
  > = DecorateFunctionParam<Ctx, A, Ctx>;

  export type ContextCallbackLayer<
    Ctx extends Context,
    A extends any[] = any[],
  > = DecorateFunctionParamSingle<Ctx, ContextFunction<Ctx, void>, A>;

  export type TopLevelActionDef<
    Ctx extends Context,
    A extends any[] = any[],
  > = DecorateFunctionParamSingle<Ctx, any, A>;
}

const ThirdEyeSym = Symbol('ThirdEyeSym');

export class Registrar<Ctx extends Context> {
  metadata = new MetadataSetter<
    Registrar.MetadataMap,
    Registrar.MetadataArrayMap
  >();
  reflector = new Reflector<
    Registrar.MetadataMap,
    Registrar.MetadataArrayMap
  >();
  constructor(public contextClass: ClassType<Ctx>) {}

  aspect<T>(obj: T, view: Record<any, any> = {}): RegistrarAspect<Ctx, T> {
    return new RegistrarAspect(this, obj, view);
  }

  decorateMethod<A extends any[], F extends (...args: any[]) => any>(
    type: string,
    action: Registrar.MethodResolver<Ctx, A, F>,
  ) {
    return (...args: A): TypedMethodDecorator<F> =>
      this.metadata.set(
        'CordisRegister',
        new RegisterMeta(action, args, { type }),
        'CordisRegisterKeys',
      );
  }

  decorateTransformer<A extends any[]>(
    transformer: Registrar.ContextTransformer<Ctx, A>,
  ) {
    return (...args: A): ClassDecorator & MethodDecorator =>
      this.metadata.append(
        'CordisContextTransformer',
        new RegisterMeta(transformer, args),
      );
  }

  decorateTopLevelAction<A extends any[]>(
    action: Registrar.TopLevelActionDef<Ctx, A>,
  ) {
    return (...args: A): ClassDecorator =>
      this.metadata.append(
        'CordisTopLevelAction',
        new RegisterMeta(action, args),
      );
  }

  decorateContextLayer<A extends any[]>(
    action: Registrar.ContextCallbackLayer<Ctx, A>,
  ) {
    return (...args: A): ClassDecorator & MethodDecorator =>
      this.metadata.append(
        'CordisContextLayers',
        new RegisterMeta(action, args),
      );
  }

  private getFork(obj: any) {
    const fork = this.reflector.get('CordisPluginFork', obj);
    if (!fork) {
      return;
    }
    return this.plugin()(fork);
  }

  afterPluginMethodRegistration(result: Registrar.RegisterResult<Ctx, any>) {
    // for override
  }

  plugin<T>(
    options?: PluginRegistrar.PluginRegistrationOptions<Ctx, T>,
  ): <C extends PluginRegistrar.PluginClass<Ctx, T>>(
    plugin: C,
  ) => C & PluginRegistrar.PluginRegistrationOptions<Ctx, T>;
  plugin<T>(options: PluginRegistrar.PluginRegistrationOptions<Ctx, T> = {}) {
    // eslint-disable-next-line @typescript-eslint/no-this-alias
    const _this = this;
    const reflector = this.reflector;
    return <
      C extends {
        new (...args: any[]): any;
      },
    >(
      originalClass: C,
    ) => {
      if (options.name) {
        _this.pluginDecorators().PluginName(options.name)(originalClass);
      }
      if (options.schema) {
        _this.pluginDecorators().PluginSchema(options.schema)(originalClass);
      }
      if (options.using) {
        _this.scopeDecorators().UsingService(...options.using)(originalClass);
      }
      if (originalClass[ThirdEyeSym]) {
        return originalClass;
      }
      const newClass = class
        extends originalClass
        implements PluginRegistrar.PluginMeta<Ctx>
      {
        static get Config() {
          const schemaType = reflector.get(
            'CordisPluginPredefineSchema',
            newClass,
          );
          return schemaType ? SchemaClass(schemaType) : undefined;
        }

        static get using() {
          let list = reflector.getArray('CordisPluginUsing', newClass);
          const fork = _this.getFork(newClass);
          if (fork) {
            list = [...list, ...fork.using];
          }
          return _.uniq(list);
        }

        static get reusable() {
          return reflector.get('CordisPluginReusable', newClass);
        }

        __ctx: Ctx;
        __config: T;
        __pluginOptions: PluginRegistrar.PluginRegistrationOptions<Ctx, T>;
        __registrar: RegistrarAspect<Ctx>;
        __promisesToWaitFor: Promise<any>[];
        __disposables: (() => void)[];

        _handleSystemInjections() {
          const injectKeys = reflector.getArray('CordisPluginSystemKeys', this);
          for (const key of injectKeys) {
            const valueFunction = reflector.get(
              'CordisPluginSystem',
              this,
              key,
            );
            if (!valueFunction) {
              continue;
            }
            Object.defineProperty(this, key, {
              configurable: true,
              enumerable: true,
              get: () => valueFunction(this, newClass),
            });
          }
        }

        _handleServiceInjections() {
          const injectKeys = reflector.getArray('CordisPluginInjectKeys', this);
          for (const key of injectKeys) {
            const name = reflector.get('CordisPluginInject', this, key);
            if (!name) {
              continue;
            }
            Object.defineProperty(this, key, {
              enumerable: true,
              configurable: true,
              get: () => {
                return this.__ctx[name];
              },
              set: (val: any) => {
                // eslint-disable-next-line @typescript-eslint/ban-ts-comment
                // @ts-ignore
                this.__ctx[name] = val;
              },
            });
          }
        }

        _registerDeclarations() {
          this.__registrar.register(this.__ctx).subscribe({
            next: (v) => {
              if (!v) {
                return;
              }
              const mayBePromise = v.result;
              if (mayBePromise instanceof Promise) {
                this.__promisesToWaitFor.push(mayBePromise as Promise<any>);
              }
              _this.afterPluginMethodRegistration(v);
            },
          });
        }

        _getProvidingServices() {
          return reflector.getArray('CordisPluginProvide', this);
        }

        _handleServiceProvide(immediate: boolean) {
          const providingServices = this._getProvidingServices().filter(
            (serviceDef) => !serviceDef.immediate === !immediate,
          );
          for (const key of providingServices) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            this.__ctx[key.serviceName] = this as any;
          }
        }

        _initializeFork() {
          const fork = _this.getFork(this);
          if (!fork) {
            return;
          }
          this.__ctx.on('fork', (ctx, options) => {
            ctx.__parent = this;
            const instance = new fork(ctx as Ctx, options);
            ctx.on('dispose', () => {
              if (typeof this.onForkDisconnect === 'function') {
                this.onForkDisconnect(instance);
              }
              delete ctx.__parent;
            });
            if (typeof this.onFork === 'function') {
              this.onFork(instance);
            }
          });
        }

        _uninstallServiceProvide() {
          const providingServices = this._getProvidingServices();
          for (const key of providingServices) {
            if (this.__ctx[key.serviceName] === (this as never)) {
              // eslint-disable-next-line @typescript-eslint/ban-ts-comment
              // @ts-ignore
              this.__ctx[key.serviceName] = null;
            }
          }
        }

        _registerAfterInit() {
          this.__ctx.on('ready', async () => {
            if (this.__promisesToWaitFor.length) {
              await Promise.all(this.__promisesToWaitFor);
              this.__promisesToWaitFor = [];
            }
            if (typeof this.onConnect === 'function') {
              await this.onConnect();
            }
            this._handleServiceProvide(false);
          });
          this.__ctx.on('dispose', async () => {
            this._uninstallServiceProvide();
            if (typeof this.onDisconnect === 'function') {
              await this.onDisconnect();
            }
            this.__disposables.forEach((dispose) => dispose());
            delete this.__ctx;
            delete this.__config;
            delete this.__pluginOptions;
            delete this.__registrar;
            delete this.__promisesToWaitFor;
            delete this.__disposables;
          });
        }

        _initializePluginClass() {
          this._handleSystemInjections();
          this._handleServiceInjections();
          this.__registrar.performTopActions(this.__ctx);
          this._registerDeclarations();
          if (typeof this.onApply === 'function') {
            this.onApply();
          }
          this._handleServiceProvide(true);
          this._initializeFork();
          this._registerAfterInit();
        }

        constructor(...args: any[]) {
          const originalCtx: Ctx = args[0];
          const config = args[1];
          const ctx = _this
            .aspect(newClass, config)
            .getScopeContext(originalCtx);
          super(ctx, config, ...args.slice(2));
          this.__ctx = ctx;
          this.__config = config;
          this.__pluginOptions = options;
          this.__registrar = _this.aspect(this, config);
          this.__promisesToWaitFor = [];
          this.__disposables = [];
          this._initializePluginClass();
        }
      };
      Object.defineProperty(newClass, 'name', {
        enumerable: true,
        configurable: true,
        get: () => {
          const nameFromMeta = reflector.get(
            'CordisPluginPredefineName',
            newClass,
          );
          if (nameFromMeta) {
            return nameFromMeta;
          }
          const nameFromFork = _this.getFork(newClass)?.name;
          if (nameFromFork) {
            return nameFromFork;
          }
          return originalClass.name;
        },
      });
      newClass[ThirdEyeSym] = true;
      return newClass;
    };
  }

  methodDecorators() {
    return {
      UseEvent: this.decorateMethod(
        'on',
        (ctx, fun, event: keyof GetEvents<Ctx>, prepend?: boolean) =>
          ctx.on(event as any, fun, prepend),
      ),
      UsePlugin: this.decorateMethod(
        'plugin',
        (
          ctx,
          fun: (
            ...args: any[]
          ) => Awaitable<PluginRegistrar.PluginDefinition<Ctx>>,
        ) => {
          const result = fun();
          const register = (def: PluginRegistrar.PluginDefinition<Ctx>) => {
            if (!def) {
              return;
            }
            return ctx.plugin(def.plugin, def.options);
          };
          if (result instanceof Promise) {
            return result.then(register);
          } else {
            return register(result);
          }
        },
      ),
      Apply: this.decorateMethod('apply', (ctx, fun) => fun()),
    };
  }

  scopeDecorators() {
    return {
      Isolate: (...services: string[]) =>
        this.decorateTransformer((ctx, r) => ctx.isolate(r(services))),
      UsingService: (
        ...services: string[]
      ): ClassDecorator & MethodDecorator => {
        return (obj, key?) => {
          if (!key) {
            services.forEach((service) =>
              this.metadata.appendUnique('CordisPluginUsing', service)(obj),
            );
          } else {
            const dec = this.decorateContextLayer((ctx, cb) =>
              ctx.using(services, cb),
            );
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            dec()(obj, key);
          }
        };
      },
    };
  }

  pluginDecorators() {
    const InjectSystem = (fun: PluginRegistrar.SystemInjectFun<Ctx>) =>
      this.metadata.set('CordisPluginSystem', fun, 'CordisPluginSystemKeys');
    return {
      PluginName: (name: string) =>
        this.metadata.set('CordisPluginPredefineName', name),
      PluginSchema: (schema: Schema | ClassType<any>) =>
        this.metadata.set('CordisPluginPredefineSchema', schema),
      Reusable: (reusable = true) =>
        this.metadata.set('CordisPluginReusable', reusable),
      Fork: (fork: PluginRegistrar.PluginClass<Ctx>) =>
        this.metadata.set('CordisPluginFork', fork),
      If: <T>(
        func: Condition<boolean, T, [Record<string, any>]>,
      ): MethodDecorator =>
        this.metadata.append('CordisControl', {
          type: 'if',
          condition: func,
        }),
      For: <T>(
        func: Condition<
          Iterable<Record<string, any>>,
          T,
          [Record<string, any>]
        >,
      ): MethodDecorator =>
        this.metadata.append('CordisControl', {
          type: 'for',
          condition: func,
        }),
      Provide: (
        name: string,
        options?: Registrar.ProvideOptions,
      ): ClassDecorator => {
        Context.service(name, options);
        return this.metadata.append('CordisPluginProvide', {
          ...options,
          serviceName: name,
        });
      },
      Inject: (...args: [(string | boolean)?, boolean?]): PropertyDecorator => {
        let name: string;
        let addUsing = false;
        if (args.length === 1) {
          if (typeof args[0] === 'boolean') {
            addUsing = args[0];
          } else {
            name = args[0];
          }
        } else if (args.length >= 2) {
          name = args[0] as string;
          addUsing = args[1];
        }
        return (obj, key) => {
          const serviceName = name || (key as string);
          if (addUsing) {
            this.metadata.appendUnique(
              'CordisPluginUsing',
              serviceName,
            )(obj.constructor);
          }
          const dec = this.metadata.set(
            'CordisPluginInject',
            serviceName,
            'CordisPluginInjectKeys',
          );
          return dec(obj, key);
        };
      },
      InjectSystem,
      InjectContext: () => InjectSystem((obj) => obj.__ctx),
      InjectConfig: () => InjectSystem((obj) => obj.__config),
      InjectParent: () => InjectSystem((obj) => obj.__ctx.__parent),
      Caller: () =>
        InjectSystem((obj) => {
          const targetCtx: Context = obj[Context.current] || obj.__ctx;
          return targetCtx;
        }),
      DefinePlugin: <T>(
        options?: PluginRegistrar.PluginRegistrationOptions<Ctx, T>,
      ) => this.plugin(options),
    };
  }

  transformContext<_Ctx extends Ctx>(
    ctx: _Ctx,
    filters: RegisterMeta<Registrar.ContextTransformer<Ctx>>[],
    view: any,
  ) {
    let targetCtx = ctx;
    for (const fun of filters) {
      // eslint-disable-next-line @typescript-eslint/ban-ts-comment
      // @ts-ignore
      targetCtx = fun.run(view, ctx) || targetCtx;
    }
    return targetCtx;
  }

  starterPluginFactory() {
    return <C>(config?: ClassType<C>) => {
      const plugin = class StarterPluginBase extends BasePlugin<Ctx, C> {};
      if (config) {
        this.pluginDecorators().PluginSchema(config)(plugin);
      }
      return plugin;
    };
  }
}

export const defaultRegistrar = new Registrar(Context);

export class BasePlugin<Ctx extends Context, C, PC = PartialDeep<C>> {
  constructor(public ctx: Ctx, config: PC) {}

  @defaultRegistrar.pluginDecorators().InjectConfig()
  config: C;
}
