import { Inject, Provide } from '../src/decorators';
import { DefinePlugin, StarterPlugin } from './utility/decorators';
import { LifecycleEvents } from '../src/plugin-def';
import { Context } from 'cordis';

declare module 'cordis' {
  // eslint-disable-next-line @typescript-eslint/no-namespace
  interface Context {
    immediateDependency: ImmediateDependency;
    nonImmediateDependency: NonImmediateDependency;
    myPlugin: TestingBase;
  }
}

@Provide('immediateDependency', { immediate: true })
@DefinePlugin()
class ImmediateDependency extends StarterPlugin() {}

@Provide('nonImmediateDependency')
@DefinePlugin()
class NonImmediateDependency extends StarterPlugin() {}

@Provide('myPlugin', { immediate: true })
@DefinePlugin()
class TestingBase extends StarterPlugin() implements LifecycleEvents {
  onApply() {
    this.applied = true;
  }

  onConnect() {
    this.connected = true;
  }

  onDisconnect() {
    this.disconnected = true;
  }

  applied = false;
  connected = false;
  disconnected = false;
}

class MyPlugin extends TestingBase {}

class MyPlugin2 extends TestingBase {
  @Inject(true)
  immediateDependency: ImmediateDependency;
}

class MyPlugin3 extends TestingBase {
  @Inject(true)
  nonImmediateDependency: NonImmediateDependency;
}

async function RunApplyTest(app: Context, plugin: any) {
  app.plugin(plugin);
  await app.start();
  const myPlugin = app.myPlugin;
  expect(myPlugin.applied).toBe(true);
  expect(myPlugin.connected).toBe(true);
  expect(myPlugin.disconnected).toBe(false);
  app.dispose(plugin);
  expect(myPlugin.disconnected).toBe(true);
  expect(app.immediateDependency).toBeDefined();
  expect(app.nonImmediateDependency).toBeDefined();
  await app.stop();
}

describe('Apply and Connect in koishi-thirdeye', () => {
  let app: Context;
  beforeEach(() => {
    app = new Context();
    app.plugin(ImmediateDependency);
    app.plugin(NonImmediateDependency);
  });

  it('should be applied and connected', async () => {
    await RunApplyTest(app, MyPlugin);
  });
  it('should be applied and connected with immediate dependency', async () => {
    await RunApplyTest(app, MyPlugin2);
  });
  it('should be applied and connected with non-immediate dependency', async () => {
    await RunApplyTest(app, MyPlugin3);
  });
});
