import { Context } from 'cordis';
import { Registrar } from './registrar';
import { generateRenderer, renderObject } from './utility/render-object';
import { extractObjectMethod } from './utility/utility';
import { ControlType } from './def';
import {
  from,
  mergeMap,
  Observable,
  ObservableInput,
  ObservedValueOf,
  of,
} from 'rxjs';
import _ from 'lodash';

type RecursiveUnwrapObservable<T> = T extends ObservableInput<any>
  ? RecursiveUnwrapObservable<ObservedValueOf<T>>
  : T;

export class RegistrarAspect<Ctx extends Context, T = any> {
  constructor(
    private registrar: Registrar<Ctx>,
    private obj: T,
    private view: Record<any, any> = {},
  ) {}

  getAllFieldsToRegister() {
    const arr = this.registrar.reflector
      .getArray('CordisRegisterKeys', this.obj)
      .filter((k) => typeof k === 'string');
    return arr as (keyof T & string)[];
  }

  getScopeContext(
    ctx: Ctx,
    key?: keyof T & string,
    extraView: Record<any, any> = {},
    autoScope = true,
  ) {
    if (key && autoScope) {
      ctx = this.getScopeContext(ctx);
    }
    const contextFilters = this.registrar.reflector.getArray(
      'CordisContextTransformer',
      this.obj,
      key,
    );
    const r = generateRenderer({ ...this.view, ...extraView });
    return this.registrar.transformContext(ctx, contextFilters, r);
  }

  registerMethod(
    ctx: Ctx,
    key: keyof T & string,
    extraView: Record<any, any> = {},
  ): Registrar.RegisterResult<Ctx, T> {
    const data = this.registrar.reflector.get('CordisRegister', this.obj, key);
    if (!data) return;
    const specificCtx = this.getScopeContext(ctx, key, extraView, false);
    const result = data.action(
      specificCtx,
      extractObjectMethod(this.obj, key),
      ...renderObject(data.args, { ...this.view, ...extraView }),
    );
    return { ...data, key: key, result, ctx: specificCtx };
  }

  private registerWithLoopControl(
    ctx: Ctx,
    key: keyof T & string,
    stack: ControlType[],
    existing: Record<string, any> = {},
  ): Registrar.RegisterResult<Ctx, T>[] {
    if (!stack.length) {
      const result = this.registerMethod(ctx, key, existing);
      return result ? [result] : [];
    }
    const rest = [...stack];
    const control = rest.pop();

    switch (control.type) {
      case 'if':
        if (!(control as ControlType<'if'>).condition(this.obj, existing))
          return [];
        return this.registerWithLoopControl(ctx, key, rest, existing);
      case 'for':
        return Array.from(
          (control as ControlType<'for'>).condition(this.obj, existing),
        ).flatMap((item) =>
          this.registerWithLoopControl(ctx, key, rest, {
            ...existing,
            ...item,
          }),
        );
    }
  }

  private runLayersWith<R extends ObservableInput<any>>(
    ctx: Ctx,
    cb: Registrar.ContextFunction<Ctx, R>,
    layers: Registrar.ContextCallbackLayer<Ctx>[],
  ): Observable<ObservedValueOf<R>> {
    const rest = [...layers];
    const layer = rest.pop();
    if (!layer) {
      return from(cb(ctx));
    }
    return new Observable((subscriber) => {
      layer(ctx, async (nextCtx) => {
        if (!rest.length) {
          const result = cb(nextCtx);
          if (result) {
            const tmpObs = from(result);
            tmpObs.subscribe({
              next: (v) => subscriber.next(v),
              // no error
              // no complete
            });
          }
        } else {
          this.runLayersWith(nextCtx, cb, rest).subscribe(subscriber);
        }
      });
    });
  }

  private runLayers<R extends ObservableInput<any>>(
    ctx: Ctx,
    cb: Registrar.ContextFunction<Ctx, R>,
    key?: keyof T,
  ) {
    const layers = this.registrar.reflector.getArray(
      'CordisContextLayers',
      this.obj,
      key as string,
    );
    return this.runLayersWith(ctx, cb, layers);
  }

  registerFor(ctx: Ctx, key: keyof T & string) {
    const stack = this.registrar.reflector.getArray(
      'CordisControl',
      this.obj,
      key,
    );
    return this.runLayers(
      ctx,
      (innerCtx) => this.registerWithLoopControl(innerCtx, key, stack),
      key,
    );
  }

  register(ctx: Ctx) {
    const keys = this.getAllFieldsToRegister();
    return this.runLayers(ctx, (innerCtx) =>
      keys.map((key) => this.registerFor(innerCtx, key)),
    ).pipe(mergeMap((v) => v));
  }

  performTopActions(
    ctx: Ctx,
    autoScope = false,
    extraView: Record<any, any> = {},
  ) {
    if (autoScope) {
      ctx = this.getScopeContext(ctx);
    }
    const actions = _.uniq(
      this.registrar.reflector.getArray('CordisTopLevelAction', this.obj),
    );
    const renderer = generateRenderer({ ...this.view, ...extraView });
    actions.forEach((action) => action(ctx, this.obj, renderer));
  }
}
