import { RegisterSchema, SchemaProperty } from 'schemastery-gen';
import { DefinePlugin, StarterPlugin } from './utility/decorators';
import { ParentPluginMap } from '../src/utility/parent-plugin';
import { Prop } from '../src/def';
import { Apply, InjectParent, Provide, Reusable } from '../src/decorators';
import { Context } from 'cordis';

declare module 'cordis' {
  interface Context {
    forkTest: MyPlugin;
  }
}

@RegisterSchema()
class Config {
  @SchemaProperty()
  name: string;

  getName() {
    return this.name;
  }
}

@DefinePlugin()
class ChildPlugin extends StarterPlugin(Config) {
  @InjectParent()
  parent: Prop<MyPlugin>;

  @Apply()
  increase() {
    this.parent.loadCount++;
    // console.log('fork loaded: ', this.parent.loadCount);
  }
}

@Provide('forkTest', { immediate: true })
@DefinePlugin()
class MyPlugin extends ParentPluginMap(ChildPlugin, (p) => p.config.getName()) {
  loadCount = 0;
  isParent = true;

  @Apply()
  onLoad() {
    // console.log('load', this.config);
  }
}

@Reusable()
@DefinePlugin()
class MyReusablePlugin extends StarterPlugin(Config) {
  @Apply()
  onLoad() {
    this.ctx.root['count']++;
  }
}

describe('Fork', () => {
  let app: Context;
  beforeEach(async () => {
    app = new Context();
    await app.start();
    app['count'] = 0;
  });

  it('should fork a plugin', async () => {
    // console.log('before 1');
    app.plugin(MyPlugin, { name: 'a' });
    // console.log('after 1: ' + app.forkTest.loadCount);
    const myPlugin = app.forkTest;
    expect(app.forkTest.config.getName()).toEqual('a');
    expect(app.forkTest.instances.get('a').config.getName()).toEqual('a');
    // console.log(myPlugin.instances.get('a').parent);
    // console.log(myPlugin);
    expect(myPlugin.instances.get('a').parent === myPlugin).toBe(true);
    expect(app.forkTest.instances.get('b')).toBeUndefined();
    expect(app.forkTest.loadCount).toBe(1);

    // console.log('before 2: ' + app.forkTest.loadCount);
    app.plugin(MyPlugin, { name: 'b' });
    // console.log('after 2: ' + app.forkTest.loadCount);
    expect(app.forkTest.instances.get('b').config.getName()).toEqual('b');
    // console.log(myPlugin.instances.get('b').parent);
    // console.log(myPlugin);
    expect(myPlugin.instances.get('b').parent === myPlugin).toBe(true);
    expect(app.forkTest.loadCount).toBe(2);

    // console.log('before 3: ' + app.forkTest.loadCount);
    app.plugin(MyPlugin, { name: 'c' });
    // console.log('after 3: ' + app.forkTest.loadCount);
    expect(app.forkTest.instances.get('c').config.getName()).toEqual('c');
    // console.log(myPlugin.instances.get('c').parent);
    // console.log(myPlugin);
    expect(myPlugin.instances.get('c').parent === myPlugin).toBe(true);
    expect(app.forkTest.loadCount).toBe(3);
  });

  it('it should work on reusable', async () => {
    expect(app['count']).toBe(0);
    app.plugin(MyReusablePlugin, { name: 'a' });
    expect(app['count']).toBe(1);
    app.plugin(MyReusablePlugin, { name: 'b' });
    expect(app['count']).toBe(2);
  });
});
