import { Inject, Provide, UsingService } from '../src/decorators';
import { DefinePlugin, StarterPlugin, UseEvent } from './utility/decorators';
import { Registrar } from '../src/registrar';
import { Context } from 'cordis';

declare module 'cordis' {
  interface Context {
    myProvider: MyProvider;
    myEagerProvider: MyEagerProvider;
    myConsumer: MyConsumer;
    myUsingConsumer: MyUsingConsumer;
    myPartialConsumer: MyPartialConsumer;
    dummyProvider: any;
  }

  interface Events {
    'pang'(message: string): Promise<string>;
    'pong'(message: string): Promise<string>;
  }
}

@Provide('myProvider')
@DefinePlugin()
class MyProvider extends StarterPlugin() {
  ping() {
    return 'pong';
  }

  dispose() {
    return this.ctx.dispose();
  }
}

@Provide('myEagerProvider', { immediate: true })
@DefinePlugin()
class MyEagerProvider extends StarterPlugin() {
  ping() {
    return 'pong eager';
  }

  dispose() {
    return this.ctx.dispose();
  }
}

@Provide('myConsumer', { immediate: true })
@DefinePlugin()
class MyConsumer {
  @Inject()
  myProvider: MyProvider;

  @Inject()
  myEagerProvider: MyEagerProvider;

  pongResult: string;

  eagerPongResult: string;

  @UseEvent('internal/service')
  async onService(name: string) {
    if (name === 'myProvider') {
      this.pongResult = this.myProvider.ping();
    } else if (name === 'myEagerProvider') {
      this.eagerPongResult = this.myEagerProvider.ping();
    }
  }
}

@Provide('myUsingConsumer', { immediate: true })
@DefinePlugin()
class MyUsingConsumer {
  @Inject(true)
  myProvider: MyProvider;

  @Inject(true)
  myEagerProvider: MyEagerProvider;

  pongResult: string;

  eagerPongResult: string;

  @UseEvent('internal/service')
  async onService(name: string) {
    if (name === 'myProvider') {
      this.pongResult = this.myProvider.ping();
    } else if (name === 'myEagerProvider') {
      this.eagerPongResult = this.myEagerProvider.ping();
    }
  }

  emitResult: string;
}

@Provide('myPartialConsumer', { immediate: true })
@DefinePlugin()
class MyPartialConsumer {
  @Inject()
  dummyProvider: number;

  pongResult: string;

  @UsingService('dummyProvider')
  @UseEvent('pang')
  async onPang(content: string) {
    const msg = `pang: ${content}`;
    console.log(msg);
    return msg;
  }

  @UseEvent('pong')
  async onPong(content: string) {
    const msg = `pong: ${content}`;
    console.log(msg);
    return msg;
  }
}

describe('On service', () => {
  let app: Context;

  it('Should call service', async () => {
    app = new Context();
    app.plugin(MyProvider);
    app.plugin(MyEagerProvider);
    app.plugin(MyConsumer);
    expect(app.myEagerProvider).toBeDefined();
    expect(app.myEagerProvider.ping()).toBe('pong eager');
    expect(app.myProvider).toBeUndefined();
    await app.start();
    expect(app.myConsumer).toBeDefined();
    expect(app.myProvider).toBeDefined();
    // expect(app.myConsumer.eagerPongResult).toBe('pong eager');
    expect(app.myConsumer.pongResult).toBe('pong');
  });

  it('Should call service with using', async () => {
    app = new Context();
    app.plugin(MyUsingConsumer);
    expect(app.myUsingConsumer).toBeUndefined();
    app.plugin(MyProvider);
    expect(app.myUsingConsumer).toBeUndefined();
    app.plugin(MyEagerProvider);
    expect(app.myUsingConsumer).toBeUndefined();
    await app.start();
    expect(app.myUsingConsumer).toBeDefined();
    expect(app.myProvider).toBeDefined();
    expect(app.myProvider.ping()).toBe('pong');
    expect(app.myEagerProvider).toBeDefined();
    expect(app.myEagerProvider.ping()).toBe('pong eager');
    //expect(app.myUsingConsumer.eagerPongResult).toBe('pong eager');
    //expect(app.myUsingConsumer.pongResult).toBe('pong');
  });

  it('Should handle partial using deps', async () => {
    Context.service('dummyProvider');
    app = new Context();
    await app.start();
    app.plugin(MyPartialConsumer);
    expect(app.myPartialConsumer).toBeDefined();
    expect(await app.parallel('pang', 'hello')).toEqual('hello');
    expect(await app.parallel('pong', 'hello')).toEqual('pong: hello');
    app.dummyProvider = { foo: 'bar' };
    expect(await app.parallel('pang', 'hello')).toEqual('pang: hello');
    expect(await app.parallel('pong', 'hello')).toEqual('pong: hello');
    app.dummyProvider = undefined;
    expect(await app.parallel('pang', 'hi')).toEqual('hi');
    expect(await app.parallel('pong', 'hi')).toEqual('pong: hi');
    app.dummyProvider = { foo: 'baz' };
    expect(await app.parallel('pang', 'hi')).toEqual('pang: hi');
    expect(await app.parallel('pong', 'hi')).toEqual('pong: hi');
  });
});
