import 'reflect-metadata';
import { Context, Events, Fork } from 'cordis';
import { ClassType, SchemaClass } from 'schemastery-gen';
import { MetadataSetter, Reflector } from 'typed-reflector';
import type {
  Awaitable,
  Condition,
  ControlType,
  FunctionParam,
  FunctionReturn,
  ParamRenderer,
  PartialDeep,
  PickEventFunction,
  TypedMethodDecorator,
} from './def';
import { RegistrarAspect } from './registrar-aspect';
import Schema from 'schemastery';
import { PluginRegistrar } from './plugin-def';
import _ from 'lodash';

declare module 'cordis' {
  interface Context {
    __parent?: any;
  }
}

// eslint-disable-next-line @typescript-eslint/no-namespace
export namespace Registrar {
  export interface Methods<Ctx extends Context> {
    on(event: keyof Events<Ctx>, prepend?: boolean): () => boolean;
    plugin(): Awaitable<Fork<Ctx>>;
    apply(): Awaitable<any>;
  }
  export interface MethodLimitations<Ctx extends Context> {
    on: PickEventFunction<Events<Ctx>>;
    plugin(
      ...args: any[]
    ): Awaitable<PluginRegistrar.PluginDefinition<Ctx> | undefined>;
  }

  export type MethodType<Ctx extends Context> = keyof Methods<Ctx>;
  export type MethodBody<
    C extends Context,
    K extends keyof Methods<C> = keyof Methods<C>,
  > = Methods<C>[K];
  export type MethodParams<
    C extends Context,
    K extends keyof Methods<C> = keyof Methods<C>,
  > = FunctionParam<MethodBody<C, K>>;
  export type MethodReturn<
    C extends Context,
    K extends keyof Methods<C> = keyof Methods<C>,
  > = FunctionReturn<MethodBody<C, K>>;
  export type MethodClassMethodFunction<
    C extends Context,
    K extends keyof Methods<C> = keyof Methods<C>,
  > = K extends keyof MethodLimitations<C>
    ? MethodLimitations<C>[K]
    : (...args: any[]) => any;
  export type MethodResolver<
    C extends Context,
    K extends keyof Methods<C> = keyof Methods<C>,
  > = (
    ctx: C,
    fun: MethodClassMethodFunction<C, K>,
    ...args: MethodParams<C, K>
  ) => MethodReturn<C, K>;

  export interface RegisterInfo<C extends Context> {
    type: MethodType<C>;
    args: MethodParams<C>;
    action: MethodResolver<C>;
  }

  export interface RegisterResult<
    C extends Context,
    T,
    K extends keyof Methods<C> = keyof Methods<C>,
  > extends RegisterInfo<C> {
    key: keyof T & string;
    result: MethodReturn<C, K>;
    ctx: C;
  }

  export interface ProvideOptions extends Context.ServiceOptions {
    immediate?: boolean;
  }
  export interface ProvideDefinition<C extends Context> extends ProvideOptions {
    serviceName: ServiceName<C>;
  }

  export type ContextFunction<C extends Context, T> = (ctx: C) => T;
  export type RenderedContextTransformer<C extends Context> = (
    ctx: C,
    r: ParamRenderer,
  ) => C;
  export type ContextCallbackLayer<C extends Context, T = any> = (
    ctx: C,
    cb: ContextFunction<C, void>,
  ) => T;

  export type ServiceName<C extends Context> = string;

  export type TopLevelActionDef<C extends Context> = (
    ctx: C,
    obj: any,
    renderer: ParamRenderer,
  ) => void;

  export interface MetadataArrayMap<C extends Context> {
    CordisRegisterKeys: string;
    CordisContextTransformer: RenderedContextTransformer<C>;
    CordisTopLevelAction: TopLevelActionDef<C>;
    CordisContextLayers: ContextCallbackLayer<C>;
    CordisControl: ControlType;
    CordisPluginUsing: ServiceName<C>;
    CordisPluginProvide: ProvideDefinition<C>;
    CordisPluginInjectKeys: string;
    CordisPluginSystemKeys: string;
  }
  export interface MetadataMap<C extends Context> {
    CordisRegister: RegisterInfo<C>;
    CordisPluginInject: ServiceName<C>;
    CordisPluginSystem: PluginRegistrar.SystemInjectFun<C>;
    CordisPluginPredefineSchema: Schema | ClassType<any>;
    CordisPluginPredefineName: string;
    CordisPluginFork: PluginRegistrar.PluginClass<C>;
    CordisPluginReusable: boolean;
  }
}

const ThirdEyeSym = Symbol('ThirdEyeSym');

export class Registrar<Ctx extends Context> {
  metadata = new MetadataSetter<
    Registrar.MetadataMap<Ctx>,
    Registrar.MetadataArrayMap<Ctx>
  >();
  reflector = new Reflector<
    Registrar.MetadataMap<Ctx>,
    Registrar.MetadataArrayMap<Ctx>
  >();
  constructor(public contextClass: ClassType<Ctx>) {}

  aspect<T>(obj: T, view: Record<any, any> = {}): RegistrarAspect<Ctx, T> {
    return new RegistrarAspect(this, obj, view);
  }

  decorateMethod<K extends keyof Registrar.Methods<Ctx>>(
    type: K,
    action: Registrar.MethodResolver<Ctx, K>,
  ) {
    return (...args: Registrar.MethodParams<Ctx, K>) =>
      this.metadata.set(
        'CordisRegister',
        {
          type,
          args,
          action,
        },
        'CordisRegisterKeys',
      ) as TypedMethodDecorator<Registrar.MethodClassMethodFunction<Ctx, K>>;
  }

  decorateTransformer(
    transformer: Registrar.RenderedContextTransformer<Ctx>,
  ): ClassDecorator & MethodDecorator {
    return this.metadata.append('CordisContextTransformer', transformer);
  }

  decorateTopLevelAction(action: Registrar.TopLevelActionDef<Ctx>) {
    return this.metadata.append('CordisTopLevelAction', action);
  }

  decorateContextLayer(layer: Registrar.ContextCallbackLayer<Ctx>) {
    return this.metadata.append('CordisContextLayers', layer);
  }

  private getFork(obj: any) {
    const fork = this.reflector.get('CordisPluginFork', obj);
    if (!fork) {
      return;
    }
    return this.plugin()(fork);
  }

  afterPluginMethodRegistration(
    result: Registrar.RegisterResult<Ctx, any, keyof Registrar.Methods<Ctx>>,
  ) {
    // for override
  }

  plugin<T>(
    options?: PluginRegistrar.PluginRegistrationOptions<Ctx, T>,
  ): <C extends PluginRegistrar.PluginClass<Ctx, T>>(
    plugin: C,
  ) => C & PluginRegistrar.PluginRegistrationOptions<Ctx, T>;
  plugin<T>(options: PluginRegistrar.PluginRegistrationOptions<Ctx, T> = {}) {
    // eslint-disable-next-line @typescript-eslint/no-this-alias
    const _this = this;
    const reflector = this.reflector;
    return <
      C extends {
        new (...args: any[]): any;
      },
    >(
      originalClass: C,
    ) => {
      if (options.name) {
        _this.pluginDecorators().PluginName(options.name)(originalClass);
      }
      if (options.schema) {
        _this.pluginDecorators().PluginSchema(options.schema)(originalClass);
      }
      if (options.using) {
        _this.scopeDecorators().UsingService(...options.using)(originalClass);
      }
      if (originalClass[ThirdEyeSym]) {
        return originalClass;
      }
      const newClass = class
        extends originalClass
        implements PluginRegistrar.PluginMeta<Ctx>
      {
        static get Config() {
          const schemaType = reflector.get(
            'CordisPluginPredefineSchema',
            newClass,
          );
          return schemaType ? SchemaClass(schemaType) : undefined;
        }

        static get using() {
          let list = reflector.getArray('CordisPluginUsing', newClass);
          const fork = _this.getFork(newClass);
          if (fork) {
            list = [...list, ...fork.using];
          }
          return _.uniq(list);
        }

        static get reusable() {
          return reflector.get('CordisPluginReusable', newClass);
        }

        __ctx: Ctx;
        __config: T;
        __pluginOptions: PluginRegistrar.PluginRegistrationOptions<Ctx, T>;
        __registrar: RegistrarAspect<Ctx>;
        __promisesToWaitFor: Promise<any>[];
        __disposables: (() => void)[];

        _handleSystemInjections() {
          const injectKeys = reflector.getArray('CordisPluginSystemKeys', this);
          for (const key of injectKeys) {
            const valueFunction = reflector.get(
              'CordisPluginSystem',
              this,
              key,
            );
            if (!valueFunction) {
              continue;
            }
            Object.defineProperty(this, key, {
              configurable: true,
              enumerable: true,
              get: () => valueFunction(this, newClass),
            });
          }
        }

        _handleServiceInjections() {
          const injectKeys = reflector.getArray('CordisPluginInjectKeys', this);
          for (const key of injectKeys) {
            const name = reflector.get('CordisPluginInject', this, key);
            if (!name) {
              continue;
            }
            Object.defineProperty(this, key, {
              enumerable: true,
              configurable: true,
              get: () => {
                return this.__ctx[name];
              },
              set: (val: any) => {
                // eslint-disable-next-line @typescript-eslint/ban-ts-comment
                // @ts-ignore
                this.__ctx[name] = val;
              },
            });
          }
        }

        _registerDeclarations() {
          this.__registrar.register(this.__ctx).subscribe({
            next: (v) => {
              if (!v) {
                return;
              }
              const mayBePromise = v.result;
              if (mayBePromise instanceof Promise) {
                this.__promisesToWaitFor.push(mayBePromise as Promise<any>);
              }
              _this.afterPluginMethodRegistration(v);
            },
          });
        }

        _getProvidingServices() {
          return reflector.getArray('CordisPluginProvide', this);
        }

        _handleServiceProvide(immediate: boolean) {
          const providingServices = this._getProvidingServices().filter(
            (serviceDef) => !serviceDef.immediate === !immediate,
          );
          for (const key of providingServices) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            this.__ctx[key.serviceName] = this as any;
          }
        }

        _initializeFork() {
          const fork = _this.getFork(this);
          if (!fork) {
            return;
          }
          this.__ctx.on('fork', (ctx, options) => {
            ctx.__parent = this;
            const instance = new fork(ctx as Ctx, options);
            ctx.on('dispose', () => {
              if (typeof this.onForkDisconnect === 'function') {
                this.onForkDisconnect(instance);
              }
              delete ctx.__parent;
            });
            if (typeof this.onFork === 'function') {
              this.onFork(instance);
            }
          });
        }

        _uninstallServiceProvide() {
          const providingServices = this._getProvidingServices();
          for (const key of providingServices) {
            if (this.__ctx[key.serviceName] === (this as never)) {
              // eslint-disable-next-line @typescript-eslint/ban-ts-comment
              // @ts-ignore
              this.__ctx[key.serviceName] = null;
            }
          }
        }

        _registerAfterInit() {
          this.__ctx.on('ready', async () => {
            if (this.__promisesToWaitFor.length) {
              await Promise.all(this.__promisesToWaitFor);
              this.__promisesToWaitFor = [];
            }
            if (typeof this.onConnect === 'function') {
              await this.onConnect();
            }
            this._handleServiceProvide(false);
          });
          this.__ctx.on('dispose', async () => {
            this._uninstallServiceProvide();
            if (typeof this.onDisconnect === 'function') {
              await this.onDisconnect();
            }
            this.__disposables.forEach((dispose) => dispose());
            delete this.__ctx;
            delete this.__config;
            delete this.__pluginOptions;
            delete this.__registrar;
            delete this.__promisesToWaitFor;
            delete this.__disposables;
          });
        }

        _initializePluginClass() {
          this._handleSystemInjections();
          this._handleServiceInjections();
          this.__registrar.performTopActions(this.__ctx);
          this._registerDeclarations();
          if (typeof this.onApply === 'function') {
            this.onApply();
          }
          this._handleServiceProvide(true);
          this._initializeFork();
          this._registerAfterInit();
        }

        constructor(...args: any[]) {
          const originalCtx: Ctx = args[0];
          const config = args[1];
          const ctx = _this
            .aspect(newClass, config)
            .getScopeContext(originalCtx);
          super(ctx, config, ...args.slice(2));
          this.__ctx = ctx;
          this.__config = config;
          this.__pluginOptions = options;
          this.__registrar = _this.aspect(this, config);
          this.__promisesToWaitFor = [];
          this.__disposables = [];
          this._initializePluginClass();
        }
      };
      Object.defineProperty(newClass, 'name', {
        enumerable: true,
        configurable: true,
        get: () => {
          const nameFromMeta = reflector.get(
            'CordisPluginPredefineName',
            newClass,
          );
          if (nameFromMeta) {
            return nameFromMeta;
          }
          const nameFromFork = _this.getFork(newClass)?.name;
          if (nameFromFork) {
            return nameFromFork;
          }
          return originalClass.name;
        },
      });
      newClass[ThirdEyeSym] = true;
      return newClass;
    };
  }

  methodDecorators() {
    return {
      UseEvent: this.decorateMethod(
        'on',
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        (ctx, fun, event, prepend) => ctx.on(event, fun, prepend),
      ),
      UsePlugin: this.decorateMethod('plugin', (ctx, fun) => {
        const result = fun();
        const register = (def: PluginRegistrar.PluginDefinition<Ctx, any>) => {
          if (!def) {
            return;
          }
          return ctx.plugin(def.plugin, def.options);
        };
        if (result instanceof Promise) {
          return result.then(register);
        } else {
          return register(result);
        }
      }),
      Apply: this.decorateMethod('apply', (ctx, fun) => fun()),
    };
  }

  scopeDecorators() {
    return {
      Isolate: (...services: Registrar.ServiceName<Ctx>[]) =>
        this.decorateTransformer((ctx, r) => ctx.isolate(r(services))),
      UsingService: (
        ...services: Registrar.ServiceName<Ctx>[]
      ): ClassDecorator & MethodDecorator => {
        return (obj, key?) => {
          if (!key) {
            services.forEach((service) =>
              this.metadata.appendUnique('CordisPluginUsing', service)(obj),
            );
          } else {
            const dec = this.decorateContextLayer((ctx, cb) =>
              ctx.using(services, cb),
            );
            dec(obj, key);
          }
        };
      },
    };
  }

  pluginDecorators() {
    const InjectSystem = (fun: PluginRegistrar.SystemInjectFun<Ctx>) =>
      this.metadata.set('CordisPluginSystem', fun, 'CordisPluginSystemKeys');
    return {
      PluginName: (name: string) =>
        this.metadata.set('CordisPluginPredefineName', name),
      PluginSchema: (schema: Schema | ClassType<any>) =>
        this.metadata.set('CordisPluginPredefineSchema', schema),
      Reusable: (reusable = true) =>
        this.metadata.set('CordisPluginReusable', reusable),
      Fork: (fork: PluginRegistrar.PluginClass<Ctx>) =>
        this.metadata.set('CordisPluginFork', fork),
      If: <T>(
        func: Condition<boolean, T, [Record<string, any>]>,
      ): MethodDecorator =>
        this.metadata.append('CordisControl', {
          type: 'if',
          condition: func,
        }),
      For: <T>(
        func: Condition<
          Iterable<Record<string, any>>,
          T,
          [Record<string, any>]
        >,
      ): MethodDecorator =>
        this.metadata.append('CordisControl', {
          type: 'for',
          condition: func,
        }),
      Provide: (
        name: Registrar.ServiceName<Ctx>,
        options?: Registrar.ProvideOptions,
      ): ClassDecorator => {
        Context.service(name, options);
        return this.metadata.append('CordisPluginProvide', {
          ...options,
          serviceName: name,
        });
      },
      Inject: (...args: [(string | boolean)?, boolean?]): PropertyDecorator => {
        let name: string;
        let addUsing = false;
        if (args.length === 1) {
          if (typeof args[0] === 'boolean') {
            addUsing = args[0];
          } else {
            name = args[0];
          }
        } else if (args.length >= 2) {
          name = args[0] as string;
          addUsing = args[1];
        }
        return (obj, key) => {
          const serviceName = name || (key as string);
          if (addUsing) {
            this.metadata.appendUnique(
              'CordisPluginUsing',
              serviceName,
            )(obj.constructor);
          }
          const dec = this.metadata.set(
            'CordisPluginInject',
            serviceName,
            'CordisPluginInjectKeys',
          );
          return dec(obj, key);
        };
      },
      InjectSystem,
      InjectContext: () => InjectSystem((obj) => obj.__ctx),
      InjectConfig: () => InjectSystem((obj) => obj.__config),
      InjectParent: () => InjectSystem((obj) => obj.__ctx.__parent),
      Caller: () =>
        InjectSystem((obj) => {
          const targetCtx: Context = obj[Context.current] || obj.__ctx;
          return targetCtx;
        }),
      DefinePlugin: <T>(
        options?: PluginRegistrar.PluginRegistrationOptions<Ctx, T>,
      ) => this.plugin(options),
    };
  }

  transformContext(
    ctx: Ctx,
    filters: Registrar.RenderedContextTransformer<Ctx>[],
    r: ParamRenderer = (v) => v,
  ) {
    let targetCtx = ctx;
    for (const fun of filters) {
      targetCtx = fun(targetCtx, r) || targetCtx;
    }
    return targetCtx;
  }

  starterPluginFactory() {
    return <C>(config?: ClassType<C>) => {
      const plugin = class StarterPluginBase extends BasePlugin<Ctx, C> {};
      if (config) {
        this.pluginDecorators().PluginSchema(config)(plugin);
      }
      return plugin;
    };
  }
}

export const defaultRegistrar = new Registrar(Context);

export class BasePlugin<Ctx extends Context, C, PC = PartialDeep<C>> {
  constructor(public ctx: Ctx, config: PC) {}

  @defaultRegistrar.pluginDecorators().InjectConfig()
  config: C;
}
