import { CastType } from '../src/decorators';
import { wrapClass } from '../src/utility/encode-decode';

class Address {
  street: string;
  city: string;

  formatAddress() {
    return `${this.street}, ${this.city}`;
  }
}

class User {
  name: string;

  @CastType(() => Address)
  address: Address;

  @CastType(() => Date)
  createdAt: Date;

  @CastType(() => Address)
  addresses: Address[];
}

describe('wrapClass', () => {
  it('should instantiate nested classes with CastType', () => {
    const input = {
      name: 'John',
      address: {
        street: '123 St',
        city: 'NYC',
      },
      createdAt: '2023-01-01T00:00:00.000Z',
      addresses: [
        { street: 'Street 1', city: 'City 1' },
        { street: 'Street 2', city: 'City 2' },
      ],
    };

    const result = wrapClass(User, input);

    expect(result).toBeInstanceOf(User);
    expect(result.address).toBeInstanceOf(Address);
    expect(result.createdAt).toBeInstanceOf(Date);
    expect(result.addresses[0]).toBeInstanceOf(Address);
    expect(result.address.street).toBe('123 St');
    expect(result.addresses[1].city).toBe('City 2');
    expect(result.createdAt.toISOString()).toBe('2023-01-01T00:00:00.000Z');
    expect(result.address.formatAddress()).toBe('123 St, NYC');
    expect(result.addresses[0].formatAddress()).toBe('Street 1, City 1');
    expect(result.addresses[1].formatAddress()).toBe('Street 2, City 2');
  });

  it('should return the same instance if already an instance of the class', () => {
    const user = new User();
    user.name = 'Existing';
    const result = wrapClass(User, user);
    expect(result).toBe(user);
  });

  it('should handle null or undefined gracefully', () => {
    expect(wrapClass(User, null)).toBeNull();
    expect(wrapClass(User, undefined)).toBeUndefined();
  });
});
