import { BaseDriver } from '../base-driver';
import Redis from 'ioredis';
import { Redlock } from '@sesamecare-oss/redlock';
import { Awaitable, RedisDriverOptions } from '../def';
import { createPool } from 'generic-pool';
import BetterLock from 'better-lock';

export class RedisDriver extends BaseDriver {
  async createRedisClient() {
    let redis: Redis;
    if (this.options.uri) {
      redis = new Redis(this.options.uri);
    } else {
      redis = new Redis(this.options);
    }
    // await redis.connect();
    return redis;
  }

  async useTempRedisClient<T>(cb: (redis: Redis) => Awaitable<T>) {
    const redis = await this.createRedisClient();
    try {
      return await cb(redis);
    } finally {
      await redis.quit();
    }
  }

  private pool = createPool({
    create: async () => {
      const redis = await this.createRedisClient();
      return {
        redis,
        // redlock: new Redlock([redis], this.options.lock),
      };
    },
    destroy: async ({ redis }) => {
      await redis.quit();
    },
  });
  constructor(private options: RedisDriverOptions) {
    super();
  }

  override async has(baseKey: string, key: string) {
    return (
      (await this.pool.use((r) =>
        r.redis.exists(this.usingKey(baseKey, key)),
      )) !== 0
    );
  }

  override async get(baseKey: string, key: string): Promise<Buffer> {
    return this.pool.use((r) => r.redis.getBuffer(this.usingKey(baseKey, key)));
  }

  override async set(
    baseKey: string,
    key: string,
    value: Buffer,
    ttl: number,
  ): Promise<void> {
    const redisKey = this.usingKey(baseKey, key);
    await this.pool.use((r) => {
      if (ttl) {
        return r.redis.set(redisKey, value, 'PX', ttl);
      } else {
        return r.redis.set(redisKey, value);
      }
    });
  }

  override async del(baseKey: string, key: string): Promise<boolean> {
    return !!(await this.pool.use((r) =>
      r.redis.del(this.usingKey(baseKey, key)),
    ));
  }

  private originalKeys(baseKey: string, prefix = '') {
    return this.pool.use((r) =>
      r.redis.keys(this.usingKey(baseKey, `${prefix}*`)),
    );
  }

  override async keys(baseKey: string, prefix?: string): Promise<string[]> {
    const keys = await this.originalKeys(baseKey, prefix ?? '');
    return keys.map((key) => key.slice(baseKey.length + 1));
  }

  override async clear(baseKey: string, prefix?: string): Promise<void> {
    const keys = await this.originalKeys(baseKey, prefix);
    if (!keys.length) {
      return;
    }
    await this.pool.use((r) => r.redis.del(keys));
  }

  private betterLock = new BetterLock();

  override async lock<R>(keys: string[], cb: () => Promise<R>): Promise<R> {
    const run = () =>
      this.useTempRedisClient(async (redis) => {
        const redlock = new Redlock([redis], this.options.lock);
        return redlock.using(
          keys.map((key) => `${this.options.lock?.prefix || '_lock'}:${key}`),
          this.options.lock?.duration || 5000,
          cb,
        );
      });
    if (this.options?.lock?.stacked) {
      return this.betterLock.acquire(keys, run);
    } else {
      return run();
    }
  }

  override async isFree(keys: string[]): Promise<boolean> {
    const lockKeys = keys.map(
      (key) => `${this.options.lock?.prefix || '_lock'}:${key}`,
    );
    return (await this.pool.use((r) => r.redis.exists(...lockKeys))) === 0;
  }

  quitted = false;

  async destroy() {
    this.quitted = true;
    [...this.waitingBlockingProms.values()].forEach((resolve) => resolve());
    await this.pool.drain();
  }

  private getQueueKey(key: string) {
    return `${this.options.queueKey || '_queue'}:${key}`;
  }

  private getQueueBackupKey(key: string) {
    return `${this.options.queueBackupKey || '_queue_backup'}:${key}`;
  }

  async queueLength(key: string): Promise<number> {
    const _key = this.getQueueKey(key);
    return this.pool.use((r) => r.redis.llen(_key));
  }

  async queueItems(key: string): Promise<Buffer[]> {
    const _key = this.getQueueKey(key);
    return this.pool.use((r) => r.redis.lrangeBuffer(_key, 0, -1));
  }

  override async queueAdd(
    key: string,
    value: Buffer,
    prior?: boolean,
  ): Promise<void> {
    const _key = this.getQueueKey(key);
    await this.pool.use(async (r) => {
      if (prior) {
        await r.redis.lpush(_key, value);
      } else {
        await r.redis.rpush(_key, value);
      }
    });
  }

  override async queueGather(key: string): Promise<Buffer> {
    const _key = this.getQueueKey(key);
    const backupKey = this.getQueueBackupKey(key);
    const value = await this.pool.use((r) =>
      r.redis.lmoveBuffer(_key, backupKey, 'LEFT', 'RIGHT'),
    );
    return value || undefined;
  }

  private waitingBlockingProms = new Map<Promise<Buffer>, () => void>();

  override async queueGatherBlocking(key: string): Promise<Buffer> {
    if (this.quitted) return;
    const _key = this.getQueueKey(key);
    const backupKey = this.getQueueBackupKey(key);
    const res = await this.useTempRedisClient(async (redisClient) => {
      try {
        const valueProm = redisClient.blmoveBuffer(
          _key,
          backupKey,
          'LEFT',
          'RIGHT',
          0,
        );
        const exitProm = new Promise<void>((resolve) => {
          this.waitingBlockingProms.set(valueProm, resolve);
        });
        const value = await Promise.race([valueProm, exitProm]);
        this.waitingBlockingProms.delete(valueProm);
        if (value) return value;
      } catch (e) {}
    });
    return res || this.queueGatherBlocking(key);
  }

  async queueAck(key: string, value: Buffer): Promise<void> {
    if (this.quitted) return;
    const backupKey = this.getQueueBackupKey(key);
    await this.pool.use((r) => r.redis.lrem(backupKey, 1, value));
  }

  async queueResume(
    key: string,
    value: Buffer,
    prior?: boolean,
  ): Promise<void> {
    if (this.quitted) return;
    const _key = this.getQueueKey(key);
    const backupKey = this.getQueueBackupKey(key);
    await this.pool.use(async (r) => {
      if (!(await r.redis.lrem(backupKey, 1, value))) return;
      if (prior) {
        await r.redis.lpush(_key, value);
      } else {
        await r.redis.rpush(_key, value);
      }
    });
  }

  async queueResumeAll(key: string, prior?: boolean): Promise<void> {
    if (this.quitted) return;
    const _key = this.getQueueKey(key);
    const backupKey = this.getQueueBackupKey(key);
    await this.pool.use(async (r) => {
      const values = await r.redis.lrangeBuffer(backupKey, 0, -1);
      const commands = r.redis.multi().del(backupKey);
      if (prior) {
        values.reverse();
        values.forEach((value) => commands.lpush(_key, value));
      } else {
        values.forEach((value) => commands.rpush(_key, value));
      }
      await commands.exec();
    });
  }

  async queueClear(key: string): Promise<void> {
    const _key = this.getQueueKey(key);
    await this.pool.use((r) => r.redis.del(_key));
  }
}
