import { BaseDriver } from './base-driver';
import { AnyClass, AragamiOptions, Awaitable, ClassType } from './def';
import { RedisDriver } from './drivers/redis';
import { MemoryDriver } from './drivers/memory';
import { reflector } from './metadata';
import { encode, decode } from 'encoded-buffer';

import { makeArray, MayBeArray } from './utility/utility';
import { PartialDeep } from './utility/partial-deep';
import { wrapClass } from './utility/encode-decode';

export class Aragami {
  readonly driver: BaseDriver;

  constructor(private options: AragamiOptions = {}) {
    this.driver = options.redis
      ? new RedisDriver(options.redis)
      : new MemoryDriver();
  }

  private getBaseKey(o: string | any): string {
    if (typeof o === 'string') {
      return o;
    }
    const keyFromMetadata = reflector.get('AragamiCachePrefix', o);
    if (keyFromMetadata) {
      return keyFromMetadata;
    }
    const keyFromConstructor =
      typeof o === 'function' ? o.name : o.constructor?.name;
    if (keyFromConstructor) {
      return keyFromConstructor;
    }
    return 'default';
  }

  private async getKey(o: any, prototype?: AnyClass, fallback?: string) {
    if (typeof o === 'string') {
      return o;
    }
    if (prototype) {
      o = wrapClass(prototype, o);
    }
    const keyTransformer = reflector.get('AragamiCacheKey', o);
    if (!keyTransformer) {
      if (fallback) {
        return fallback;
      }
      throw new Error(`No key metadata found for ${o.constructor.name}`);
    }
    return await keyTransformer(o);
  }

  private getTTL(o: any) {
    return reflector.get('AragamiCacheTTL', o) ?? this.options.defaultTTL ?? 0;
  }

  private encode(o: any) {
    return encode(o);
  }

  private decode<T>(cl: ClassType<T>, value: Buffer) {
    return wrapClass(cl, decode(value)[0]);
  }

  async get<T>(cl: ClassType<T>, key: string) {
    const value = await this.driver.get(this.getBaseKey(cl), key);
    if (!value) {
      return;
    }
    return this.decode(cl, value);
  }

  async set<T>(
    o: T,
    options?: { ttl?: number; key?: string; prototype?: ClassType<T> },
  ): Promise<T>;
  async set<T>(
    prototype: ClassType<T>,
    o: PartialDeep<T>,
    options?: { ttl?: number; key?: string },
  ): Promise<T>;
  async set<T>(...args: any[]) {
    let prototype: ClassType<T>;
    let o: T;
    let options: { ttl?: number; key?: string; prototype?: ClassType<T> };
    const firstArg = args[0];
    if (typeof firstArg === 'function') {
      prototype = firstArg;
      o = args[1];
      options = args[2] || {};
    } else {
      o = firstArg;
      options = args[1] || {};
      prototype = options.prototype;
    }
    if (!o) {
      return o;
    }
    if (prototype) {
      o = wrapClass(prototype, o);
    }
    const buf = this.encode(o);
    await this.driver.set(
      this.getBaseKey(o),
      options.key || (await this.getKey(o)),
      buf,
      options.ttl ?? this.getTTL(o),
    );
    return o;
  }

  async has(base: AnyClass | string, key: string): Promise<boolean>;
  async has(base: any): Promise<boolean>;
  async has(base: any, key?: string) {
    return this.driver.has(
      this.getBaseKey(base),
      key || (await this.getKey(base)),
    );
  }

  async del(base: AnyClass | string, key: string): Promise<boolean>;
  async del(base: any): Promise<boolean>;
  async del(base: any, key?: string) {
    return this.driver.del(
      this.getBaseKey(base),
      key || (await this.getKey(base)),
    );
  }

  async clear(base: AnyClass | string, prefix?: string) {
    return this.driver.clear(this.getBaseKey(base), prefix);
  }

  async keys(base: AnyClass | string, prefix?: string) {
    return this.driver.keys(this.getBaseKey(base), prefix);
  }

  async values<T>(cl: ClassType<T>, prefix?: string) {
    const buffers = await this.driver.values(this.getBaseKey(cl), prefix);
    return buffers.map((buf) => this.decode(cl, buf));
  }

  async entries<T>(cl: ClassType<T>, prefix?: string): Promise<[string, T][]> {
    const entries = await this.driver.entries(this.getBaseKey(cl), prefix);
    return entries.map(([key, buf]) => [key, this.decode(cl, buf)]);
  }

  async cache<T>(
    cl: ClassType<T>,
    keyOrMeta: string | T,
    cb: () => Awaitable<T>,
  ) {
    const key = await this.getKey(keyOrMeta, cl);
    if (!key) {
      return cb();
    }
    const cachedValue = await this.get(cl, key);
    if (cachedValue != null) {
      return cachedValue;
    }
    const value = await cb();
    if (value != null) {
      await this.set(value, { key, prototype: cl });
    }
    return value;
  }

  useCache<T, A extends any[]>(
    cl: ClassType<T>,
    cb: (...args: A) => Awaitable<T>,
    keySource: (...args: A) => Awaitable<string | T>,
  ) {
    return async (...args: A): Promise<T> => {
      const keyMeta = await keySource(...args);
      return this.cache(cl, keyMeta, () => cb(...args));
    };
  }

  private async getLockKeys(o: any) {
    if (typeof o === 'string') {
      return [o];
    }
    const baseKey = this.getBaseKey(o);
    const keyTransformers = reflector.getArray('AragamiLockKeys', o);
    const actualKeys = await Promise.all(keyTransformers.map((fn) => fn(o)));
    return actualKeys
      .flatMap((mayBeKeyArray) =>
        makeArray(mayBeKeyArray).map((key) => `${baseKey}:${key}`),
      )
      .filter((s) => !!s);
  }

  async lock<R>(
    keys: MayBeArray<string | any>,
    cb: () => Awaitable<R>,
  ): Promise<R> {
    const keyMeta = makeArray(keys);
    const actualKeys = (
      await Promise.all(keyMeta.map((o) => this.getLockKeys(o)))
    ).flat();
    if (!actualKeys.length) {
      return cb();
    }
    return this.driver.lock(actualKeys, async () => await cb());
  }

  async isFree(keys: MayBeArray<string | any>) {
    const keyMeta = makeArray(keys);
    const actualKeys = (
      await Promise.all(keyMeta.map((o) => this.getLockKeys(o)))
    ).flat();
    if (!keys.length) {
      return true;
    }
    return this.driver.isFree(actualKeys);
  }

  useLock<A extends any[], R>(
    cb: (...args: A) => R,
    keySource: (...args: A) => Awaitable<MayBeArray<string | any>>,
  ) {
    return async (...args: A) => {
      const keys = await keySource(...args);
      return this.lock(keys, () => cb(...args));
    };
  }

  async destroy() {
    try {
      await this.driver.destroy();
    } catch (e) {}
  }

  async isQueueEmpty<T>(cl: ClassType<T>, key = 'default') {
    return this.driver.isQueueEmpty(this.getBaseKey(cl) + ':' + key);
  }

  async queueLength<T>(cl: ClassType<T>, key = 'default') {
    return this.driver.queueLength(this.getBaseKey(cl) + ':' + key);
  }

  async queueItems<T>(cl: ClassType<T>, key = 'default') {
    const items = await this.driver.queueItems(this.getBaseKey(cl) + ':' + key);
    return items.map((buf) => this.decode(cl, buf));
  }

  async queueAdd<T>(
    o: T,
    options?: { key?: string; prototype?: ClassType<T>; prior?: boolean },
  ): Promise<T>;
  async queueAdd<T>(
    prototype: ClassType<T>,
    o: PartialDeep<T>,
    options?: { key?: string; prior?: boolean },
  ): Promise<T>;
  async queueAdd<T>(...args: any[]) {
    let prototype: ClassType<T>;
    let o: T;
    let options: {
      ttl?: number;
      key?: string;
      prototype?: ClassType<T>;
      prior?: boolean;
    };
    const firstArg = args[0];
    if (typeof firstArg === 'function') {
      prototype = firstArg;
      o = args[1];
      options = args[2] || {};
    } else {
      o = firstArg;
      options = args[1] || {};
      prototype = options.prototype;
    }
    if (!o) {
      return o;
    }
    if (prototype) {
      o = wrapClass(prototype, o);
    }
    const buf = this.encode(o);
    const key =
      this.getBaseKey(o) +
      ':' +
      (options.key || (await this.getKey(o, undefined, 'default')));
    await this.driver.queueAdd(key, buf, options.prior);
    return o;
  }

  async queueGather<T>(prototype: ClassType<T>, key = 'default'): Promise<T> {
    const baseKey = this.getBaseKey(prototype);
    const buffer = await this.driver.queueGather(baseKey + ':' + key);
    if (!buffer) {
      return;
    }
    return this.decode(prototype, buffer);
  }

  async queueGatherBlocking<T>(
    prototype: ClassType<T>,
    key = 'default',
  ): Promise<T> {
    const baseKey = this.getBaseKey(prototype);
    const buffer = await this.driver.queueGatherBlocking(baseKey + ':' + key);
    return this.decode(prototype, buffer);
  }

  async queueClear<T>(prototype: ClassType<T>, key = 'default') {
    const baseKey = this.getBaseKey(prototype);
    await this.driver.queueClear(baseKey + ':' + key);
  }

  async queueResumeAll<T>(
    prototype: ClassType<T>,
    key = 'default',
    prior?: boolean,
  ) {
    const baseKey = this.getBaseKey(prototype);
    await this.driver.queueResumeAll(baseKey + ':' + key, prior);
  }

  async runQueueOnce<T, R>(
    prototype: ClassType<T>,
    cb: (item: T) => Awaitable<R>,
    key = 'default',
  ) {
    const baseKey = this.getBaseKey(prototype);
    const buffer = await this.driver.queueGatherBlocking(baseKey + ':' + key);
    const object = this.decode(prototype, buffer);
    try {
      const res = await cb(object);
      await this.driver.queueAck(baseKey + ':' + key, buffer);
      return res;
    } catch (e) {
      await this.driver.queueResume(baseKey + ':' + key, buffer, true);
      throw e;
    }
  }
}
