import { Metadata } from './metadata';

import { Awaitable, ClassType, TypedMethodDecorator } from './def';
import { MayBeArray } from './utility/utility';
import { Type } from 'class-transformer';
import { MergePropertyDecorators } from './utility/merge';

export const CacheTTL = (ttl: number): ClassDecorator & MethodDecorator =>
  Metadata.set('AragamiCacheTTL', ttl);

export const CachePrefix = (prefix: string): ClassDecorator =>
  Metadata.set('AragamiCachePrefix', prefix);

type ObjectFunction<T> = (o: any) => Awaitable<T>;
type ObjectAndKeyFunction<T> = (o: any, key: string) => Awaitable<T>;

const DrainKey =
  <T>(
    decoratorFactory: (cb: ObjectFunction<T>) => ClassDecorator,
    defaultValue: ObjectAndKeyFunction<T>,
  ) =>
  (
    fun: ObjectAndKeyFunction<T> = defaultValue,
  ): PropertyDecorator & TypedMethodDecorator<ObjectFunction<T>> =>
  (target: any, key: string, descriptor?: PropertyDescriptor) => {
    let cb: ObjectFunction<T>;

    if (descriptor) {
      // method decorator: descriptor.value
      if (typeof descriptor.value === 'function') {
        cb = async (o) => fun(await descriptor.value.call(o), key);
      }
      // getter/accessor decorator: descriptor.get
      else if (typeof descriptor.get === 'function') {
        cb = async (o) => fun(await descriptor.get.call(o), key);
      } else {
        // unlikely: setter-only accessor etc.
        cb = async (o) => fun(await (o as any)[key], key);
      }
    } else {
      // property decorator
      cb = (o) => fun((o as any)[key], key);
    }

    return decoratorFactory(cb)((target as any).constructor);
  };

export const CacheKey = DrainKey<string>(
  (cb) => Metadata.set('AragamiCacheKey', cb),
  (o) => o.toString(),
);
export const LockKey = DrainKey<MayBeArray<string>>(
  (cb) => Metadata.append('AragamiLockKeys', cb),
  (o, key) => `${key}_${o}`,
);

export const CastType = <T>(cb: () => ClassType<T>): PropertyDecorator =>
  MergePropertyDecorators([Metadata.set('AragamiType', cb), Type(cb)]);
