import { BaseDriver } from '../base-driver';
import Redis from 'ioredis';
import Redlock from '@nanahira/redlock';
import { RedisDriverOptions } from '../def';

export class RedisDriver extends BaseDriver {
  private readonly redis: Redis;
  private readonly redlock: Redlock;
  constructor(private options: RedisDriverOptions) {
    super();
    if (options.uri) {
      this.redis = new Redis(options.uri);
    } else {
      this.redis = new Redis(options);
    }
    this.redlock = new Redlock([this.redis], options.lock);
  }

  override async has(baseKey: string, key: string) {
    return (await this.redis.exists(this.usingKey(baseKey, key))) !== 0;
  }

  override async get(baseKey: string, key: string): Promise<Buffer> {
    return this.redis.getBuffer(this.usingKey(baseKey, key));
  }

  override async set(
    baseKey: string,
    key: string,
    value: Buffer,
    ttl: number,
  ): Promise<void> {
    const redisKey = this.usingKey(baseKey, key);
    if (ttl) {
      await this.redis.set(redisKey, value, 'PX', ttl);
    } else {
      await this.redis.set(redisKey, value);
    }
  }

  override async del(baseKey: string, key: string): Promise<boolean> {
    return !!this.redis.del(this.usingKey(baseKey, key));
  }

  private originalKeys(baseKey: string, prefix = '') {
    return this.redis.keys(this.usingKey(baseKey, `${prefix}*`));
  }

  override async keys(baseKey: string, prefix?: string): Promise<string[]> {
    const keys = await this.originalKeys(baseKey, prefix ?? '');
    return keys.map((key) => key.slice(baseKey.length + 1));
  }

  override async clear(baseKey: string, prefix?: string): Promise<void> {
    const keys = await this.originalKeys(baseKey, prefix);
    if (!keys.length) {
      return;
    }
    await this.redis.del(keys);
  }

  override lock<R>(keys: string[], cb: () => Promise<R>): Promise<R> {
    return this.redlock.using(
      keys.map((key) => `${this.options.lock?.prefix || '_lock'}:${key}`),
      this.options.lock?.duration || 5000,
      cb,
    );
  }

  async destroy() {
    await this.redis.quit();
  }
}
