import { Aragami } from '../src/aragami';
import { CacheKey, CachePrefix, CacheTTL, LockKey } from '../src/decorators';
import { WithKey, WithLockKey, WrapDecoratorBuilder } from '../src/wrappers';

describe('Aragami.', () => {
  let aragami: Aragami;

  beforeEach(() => {
    aragami = new Aragami({
      redis: { uri: 'redis://localhost:6379' },
    });
  });

  afterEach(async () => {
    await aragami.destroy();
  });

  it('Should store value', async () => {
    @CachePrefix('user')
    @CacheTTL(100)
    class User {
      name: string;
      age: number;

      @CacheKey()
      async getName() {
        return `n.${this.name}`;
      }
    }

    const user = new User();
    await aragami.clear(User);
    user.name = 'John';
    user.age = 30;
    const userSet = await aragami.set(user);
    expect(userSet).toEqual(user);
    await expect(aragami.has(user)).resolves.toBeTruthy();
    await expect(aragami.has(User, 'n.John')).resolves.toBeTruthy();
    await expect(aragami.has('user', 'n.John')).resolves.toBeTruthy();
    const tmpUser = await aragami.set(User, { name: 'Nanahira', age: 17 });
    expect(tmpUser).toEqual({ name: 'Nanahira', age: 17 });
    expect(tmpUser).toBeInstanceOf(User);
    await expect(aragami.del(tmpUser)).resolves.toBeTruthy();
    const userGet = await aragami.get(User, 'n.John');
    expect(userGet).toEqual({ name: 'John', age: 30 });
    expect(userGet).toBeInstanceOf(User);
    await expect(aragami.keys(User)).resolves.toEqual(['n.John']);
    await expect(aragami.values(User)).resolves.toEqual([
      { name: 'John', age: 30 },
    ]);
    await expect(aragami.entries(User)).resolves.toEqual([
      ['n.John', { name: 'John', age: 30 }],
    ]);
    await expect(aragami.del(User, 'n.John')).resolves.toBeTruthy();
    await expect(aragami.has(user)).resolves.toBeFalsy();

    const wrapped = aragami.useCache(
      User,
      (name: string, age: number) => {
        const user = new User();
        user.name = name;
        user.age = age;
        console.log('new', user);
        return user;
      },
      (name) => name,
    );
    await expect(wrapped('Sarah', 40)).resolves.toBeInstanceOf(User);
    await expect(aragami.has(User, 'Sarah')).resolves.toBeTruthy();
    await expect(wrapped('Sarah', 40)).resolves.toBeInstanceOf(User);
  });

  it('should expire', async () => {
    @CacheTTL(100)
    class Dress {
      @CacheKey()
      name: string;
    }

    const dress = new Dress();
    await aragami.clear(Dress);
    dress.name = 'Yuzu';
    await aragami.set(dress);
    await expect(aragami.has(dress)).resolves.toBeTruthy();
    await new Promise((resolve) => setTimeout(resolve, 90));
    await expect(aragami.has(dress)).resolves.toBeTruthy();
    await new Promise((resolve) => setTimeout(resolve, 15));
    await expect(aragami.has(dress)).resolves.toBeFalsy();
  });

  it('should lock', async () => {
    const fun = aragami.useLock(
      async (foo: string, bar: string) => {
        await new Promise((resolve) => setTimeout(resolve, 1000));
        return `${foo}.${bar}`;
      },
      (foo, bar) => [foo, bar],
    );
    await expect(fun('foo', 'bar')).resolves.toEqual('foo.bar');
  });

  it('should wrap class', async () => {
    const { UseCache, UseLock } = new WrapDecoratorBuilder(
      () => aragami,
    ).build();

    class Book {
      @CacheKey()
      @LockKey()
      title: string;
      @LockKey()
      content: string;
    }

    class MyService {
      @UseCache(Book)
      async getBook(@WithKey() title: string, content: string) {
        const book = new Book();
        book.title = title;
        book.content = content;
        console.log('got book', book);
        return book;
      }

      @UseLock()
      async saveBook(@WithLockKey() book: Book) {
        console.log('saving book', book);
        await new Promise((resolve) => setTimeout(resolve, 1000));
        console.log('saved book', book);
        return book;
      }
    }

    await aragami.clear(Book);

    const service = new MyService();
    const book = await service.getBook('foo', 'bar');
    const bookFromCache = await aragami.get(Book, 'foo');
    expect(bookFromCache).toEqual(book);
    await new Promise((resolve) => setTimeout(resolve, 20));
    const book2 = await service.getBook('foo', 'baz');
    expect(book2).toEqual({ title: 'foo', content: 'bar' });
    const savedBook = await service.saveBook(book);
    expect(savedBook).toEqual(book);
  });
});
