import LRUCache from 'lru-cache';
import { BaseDriver } from '../base-driver';
import BetterLock from 'better-lock';

export class MemoryDriver extends BaseDriver {
  private cacheMap = new Map<string, LRUCache<string, Buffer>>();
  private betterLock = new BetterLock();

  private getCacheInstance(baseKey: string) {
    if (!this.cacheMap.has(baseKey)) {
      this.cacheMap.set(
        baseKey,
        new LRUCache({
          ttl: 1,
          updateAgeOnGet: false,
          updateAgeOnHas: false,
        }),
      );
    }
    return this.cacheMap.get(baseKey);
  }

  override async get(baseKey: string, key: string): Promise<Buffer> {
    const cache = this.getCacheInstance(baseKey);
    return cache.get(key);
  }

  override async set(
    baseKey: string,
    key: string,
    value: Buffer,
    ttl: number,
  ): Promise<void> {
    const cache = this.getCacheInstance(baseKey);
    cache.set(key, value, { ttl });
  }

  override async del(baseKey: string, key: string): Promise<boolean> {
    const cache = this.getCacheInstance(baseKey);
    return cache.delete(key);
  }

  override async keys(baseKey: string): Promise<string[]> {
    const cache = this.getCacheInstance(baseKey);
    return Array.from(cache.keys());
  }

  override async clear(baseKey: string): Promise<void> {
    const cache = this.getCacheInstance(baseKey);
    cache.clear();
  }

  override lock<R>(keys: string[], cb: () => Promise<R>): Promise<R> {
    return this.betterLock.acquire(keys, cb);
  }
}
