import { BaseDriver } from './base-driver';
import { AnyClass, AragamiOptions, Awaitable, ClassType } from './def';
import { RedisDriver } from './drivers/redis';
import { MemoryDriver } from './drivers/memory';
import { reflector } from './metadata';
import { instanceToPlain, plainToInstance } from 'class-transformer';
import { encode, decode } from 'encoded-buffer';

import { makeArray, MayBeArray } from './utility/utility';
import _ from 'lodash';
import { PartialDeep } from './utility/partial-deep';

export class Aragami {
  readonly driver: BaseDriver;

  constructor(private options: AragamiOptions = {}) {
    this.driver = options.redis
      ? new RedisDriver(options.redis)
      : new MemoryDriver();
  }

  private getBaseKey(o: string | any): string {
    if (typeof o === 'string') {
      return o;
    }
    const keyFromMetadata = reflector.get('AragamiCachePrefix', o);
    if (keyFromMetadata) {
      return keyFromMetadata;
    }
    const keyFromConstructor =
      typeof o === 'function' ? o.name : o.constructor?.name;
    if (keyFromConstructor) {
      return keyFromConstructor;
    }
    return 'default';
  }

  private async getKey(o: any, prototype?: AnyClass) {
    if (typeof o === 'string') {
      return o;
    }
    if (prototype) {
      o = plainToInstance(prototype, o);
    }
    const keyTransformer = reflector.get('AragamiCacheKey', o);
    if (!keyTransformer) {
      throw new Error(`No key metadata found for ${o.constructor.name}`);
    }
    return await keyTransformer(o);
  }

  private getTTL(o: any) {
    return reflector.get('AragamiCacheTTL', o) ?? this.options.defaultTTL ?? 0;
  }

  private encode(o: any) {
    return encode(instanceToPlain(o));
  }

  private decode<T>(cl: ClassType<T>, value: Buffer) {
    return plainToInstance(cl, decode(value)[0]);
  }

  async get<T>(cl: ClassType<T>, key: string) {
    const value = await this.driver.get(this.getBaseKey(cl), key);
    if (!value) {
      return;
    }
    return this.decode(cl, value);
  }

  async set<T>(
    o: T,
    options?: { ttl?: number; key?: string; prototype?: ClassType<T> },
  ): Promise<T>;
  async set<T>(
    prototype: ClassType<T>,
    o: PartialDeep<T>,
    options?: { ttl?: number; key?: string },
  ): Promise<T>;
  async set<T>(...args: any[]) {
    let prototype: ClassType<T>;
    let o: T;
    let options: { ttl?: number; key?: string; prototype?: ClassType<T> };
    const firstArg = args[0];
    if (typeof firstArg === 'function') {
      prototype = firstArg;
      o = args[1];
      options = args[2] || {};
    } else {
      o = firstArg;
      options = args[1] || {};
      prototype = options.prototype;
    }
    if (!o) {
      return o;
    }
    if (prototype) {
      o = plainToInstance(prototype, o);
    }
    const buf = this.encode(o);
    await this.driver.set(
      this.getBaseKey(o),
      options.key || (await this.getKey(o)),
      buf,
      options.ttl ?? this.getTTL(o),
    );
    return o;
  }

  async has(base: AnyClass | string, key: string): Promise<boolean>;
  async has(base: any): Promise<boolean>;
  async has(base: any, key?: string) {
    return this.driver.has(
      this.getBaseKey(base),
      key || (await this.getKey(base)),
    );
  }

  async del(base: AnyClass | string, key: string): Promise<boolean>;
  async del(base: any): Promise<boolean>;
  async del(base: any, key?: string) {
    return this.driver.del(
      this.getBaseKey(base),
      key || (await this.getKey(base)),
    );
  }

  async clear(base: AnyClass | string) {
    return this.driver.clear(this.getBaseKey(base));
  }

  async keys(base: AnyClass | string) {
    return this.driver.keys(this.getBaseKey(base));
  }

  async values<T>(cl: ClassType<T>) {
    const buffers = await this.driver.values(this.getBaseKey(cl));
    return buffers.map((buf) => this.decode(cl, buf));
  }

  async entries<T>(cl: ClassType<T>): Promise<[string, T][]> {
    const entries = await this.driver.entries(this.getBaseKey(cl));
    return entries.map(([key, buf]) => [key, this.decode(cl, buf)]);
  }

  wrap<T, A extends any[]>(
    cl: ClassType<T>,
    cb: (...args: A) => Awaitable<T>,
    keySource: (...args: A) => Awaitable<string | T>,
  ) {
    return async (...args: A): Promise<T> => {
      const keyMeta = await keySource(...args);
      const key = await this.getKey(keyMeta, cl);
      if (!key) {
        return cb(...args);
      }
      const cachedValue = await this.get(cl, key);
      if (cachedValue) {
        return cachedValue;
      }
      const value = await cb(...args);
      if (value) {
        await this.set(value, { key, prototype: cl });
      }
      return value;
    };
  }

  private async getLockKeys(o: any) {
    if (typeof o === 'string') {
      return [o];
    }
    const baseKey = this.getBaseKey(o);
    const keyTransformers = reflector.getArray('AragamiLockKeys', o);
    const actualKeys = await Promise.all(keyTransformers.map((fn) => fn(o)));
    return _.compact(
      actualKeys.flatMap((mayBeKeyArray) =>
        makeArray(mayBeKeyArray).map((key) => `${baseKey}:${key}`),
      ),
    );
  }

  lock<A extends any[], R>(
    cb: (...args: A) => R,
    keySource: (...args: A) => Awaitable<MayBeArray<string | any>>,
  ) {
    return async (...args: A): Promise<Awaited<R>> => {
      const keyMeta = makeArray(await keySource(...args));
      const keys = (
        await Promise.all(keyMeta.map((o) => this.getLockKeys(o)))
      ).flat();
      if (!keys.length) {
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        return cb(...args);
      }
      // eslint-disable-next-line @typescript-eslint/ban-ts-comment
      // @ts-ignore
      return this.driver.lock(keys, async () => await cb(...args));
    };
  }
}
