import { BaseDriver } from '../base-driver';
import Redis from 'ioredis';
import Redlock from '@nanahira/redlock';
import { RedisDriverOptions } from '../def';
import { createPool } from 'generic-pool';

export class RedisDriver extends BaseDriver {
  async createRedisClient() {
    let redis: Redis;
    if (this.options.uri) {
      redis = new Redis(this.options.uri);
    } else {
      redis = new Redis(this.options);
    }
    // await redis.connect();
    return redis;
  }

  private pool = createPool({
    create: async () => {
      const redis = await this.createRedisClient();
      return {
        redis,
        redlock: new Redlock([redis], this.options.lock),
      };
    },
    destroy: async ({ redis }) => {
      await redis.quit();
    },
  });
  constructor(private options: RedisDriverOptions) {
    super();
  }

  override async has(baseKey: string, key: string) {
    return (
      (await this.pool.use((r) =>
        r.redis.exists(this.usingKey(baseKey, key)),
      )) !== 0
    );
  }

  override async get(baseKey: string, key: string): Promise<Buffer> {
    return this.pool.use((r) => r.redis.getBuffer(this.usingKey(baseKey, key)));
  }

  override async set(
    baseKey: string,
    key: string,
    value: Buffer,
    ttl: number,
  ): Promise<void> {
    const redisKey = this.usingKey(baseKey, key);
    await this.pool.use((r) => {
      if (ttl) {
        return r.redis.set(redisKey, value, 'PX', ttl);
      } else {
        return r.redis.set(redisKey, value);
      }
    });
  }

  override async del(baseKey: string, key: string): Promise<boolean> {
    return !!(await this.pool.use((r) =>
      r.redis.del(this.usingKey(baseKey, key)),
    ));
  }

  private originalKeys(baseKey: string, prefix = '') {
    return this.pool.use((r) =>
      r.redis.keys(this.usingKey(baseKey, `${prefix}*`)),
    );
  }

  override async keys(baseKey: string, prefix?: string): Promise<string[]> {
    const keys = await this.originalKeys(baseKey, prefix ?? '');
    return keys.map((key) => key.slice(baseKey.length + 1));
  }

  override async clear(baseKey: string, prefix?: string): Promise<void> {
    const keys = await this.originalKeys(baseKey, prefix);
    if (!keys.length) {
      return;
    }
    await this.pool.use((r) => r.redis.del(keys));
  }

  override lock<R>(keys: string[], cb: () => Promise<R>): Promise<R> {
    return this.pool.use((r) =>
      r.redlock.using(
        keys.map((key) => `${this.options.lock?.prefix || '_lock'}:${key}`),
        this.options.lock?.duration || 5000,
        cb,
      ),
    );
  }

  quitted = false;

  async destroy() {
    this.quitted = true;
    [...this.waitingBlockingProms.values()].forEach((resolve) => resolve());
    await this.pool.drain();
  }

  private getQueueKey(key: string) {
    return `${this.options.queueKey || '_queue'}:${key}`;
  }

  async queueLength(key: string): Promise<number> {
    const _key = this.getQueueKey(key);
    return this.pool.use((r) => r.redis.llen(_key));
  }

  async queueItems(key: string): Promise<Buffer[]> {
    const _key = this.getQueueKey(key);
    const items = await this.pool.use((r) => r.redis.lrangeBuffer(_key, 0, -1));
    return items.reverse();
  }

  override async queueAdd(
    key: string,
    value: Buffer,
    prior?: boolean,
  ): Promise<void> {
    const _key = this.getQueueKey(key);
    await this.pool.use(async (r) => {
      if (prior) {
        await r.redis.lpush(_key, value);
      } else {
        await r.redis.rpush(_key, value);
      }
    });
  }

  override async queueGather(key: string): Promise<Buffer> {
    const _key = this.getQueueKey(key);
    const value = await this.pool.use((r) => r.redis.rpopBuffer(_key));
    return value || undefined;
  }

  private waitingBlockingProms = new Map<
    Promise<[Buffer, Buffer]>,
    () => void
  >();

  override async queueGatherBlocking(key: string): Promise<Buffer> {
    if (this.quitted) return;
    const _key = this.getQueueKey(key);
    const redisClient = await this.createRedisClient();
    try {
      const valueProm = redisClient.brpopBuffer(_key, 0);
      const exitProm = new Promise<void>((resolve) => {
        this.waitingBlockingProms.set(valueProm, resolve);
      });
      const value = await Promise.race([valueProm, exitProm]);
      this.waitingBlockingProms.delete(valueProm);
      if (value) return value?.[1];
      //console.log('wait2');
      return await this.queueGatherBlocking(key);
    } catch (e) {
      //console.log(e);
      return await this.queueGatherBlocking(key);
    } finally {
      await redisClient.quit();
    }
  }

  async queueClear(key: string): Promise<void> {
    const _key = this.getQueueKey(key);
    await this.pool.use((r) => r.redis.del(_key));
  }
}
