import LRUCache from 'lru-cache';
import { BaseDriver } from '../base-driver';
import BetterLock from 'better-lock';

export class MemoryDriver extends BaseDriver {
  private cacheMap = new Map<string, LRUCache<string, Buffer>>();
  private betterLock = new BetterLock();

  async destroy() {
    for (const cache of this.cacheMap.values()) {
      cache.clear();
    }
    this.cacheMap.clear();
    this.queues.clear();
    [...this.blockingGathers.values()].forEach((v) =>
      v.forEach((f) => f(undefined)),
    );
    this.blockingGathers.clear();
  }

  private getCacheInstance(baseKey: string) {
    if (!this.cacheMap.has(baseKey)) {
      this.cacheMap.set(
        baseKey,
        new LRUCache({
          ttl: 1,
          updateAgeOnGet: false,
          updateAgeOnHas: false,
        }),
      );
    }
    return this.cacheMap.get(baseKey);
  }

  override async get(baseKey: string, key: string): Promise<Buffer> {
    const cache = this.getCacheInstance(baseKey);
    return cache.get(key);
  }

  override async set(
    baseKey: string,
    key: string,
    value: Buffer,
    ttl: number,
  ): Promise<void> {
    const cache = this.getCacheInstance(baseKey);
    cache.set(key, value, { ttl });
  }

  override async del(baseKey: string, key: string): Promise<boolean> {
    const cache = this.getCacheInstance(baseKey);
    return cache.delete(key);
  }

  override async keys(baseKey: string, prefix?: string): Promise<string[]> {
    const cache = this.getCacheInstance(baseKey);
    let keys = Array.from(cache.keys());
    if (prefix) {
      keys = keys.filter((key) => key.startsWith(prefix));
    }
    return keys;
  }

  override async clear(baseKey: string, prefix?: string): Promise<void> {
    const cache = this.getCacheInstance(baseKey);
    if (prefix) {
      const keys = Array.from(cache.keys());
      for (const key of keys) {
        if (key.startsWith(prefix)) {
          cache.delete(key);
        }
      }
    } else {
      cache.clear();
    }
  }

  override lock<R>(keys: string[], cb: () => Promise<R>): Promise<R> {
    return this.betterLock.acquire(keys, cb);
  }

  private queues = new Map<string, Buffer[]>();
  private blockingGathers = new Map<string, ((buf: Buffer) => void)[]>();

  override async queueAdd(
    key: string,
    value: Buffer,
    prior?: boolean,
  ): Promise<void> {
    if (this.blockingGathers.has(key)) {
      const cb = this.blockingGathers.get(key).shift();
      if (cb) {
        cb(value);
        return;
      }
    }
    if (!this.queues.has(key)) {
      this.queues.set(key, []);
    }
    const queue = this.queues.get(key);
    if (prior) {
      queue.unshift(value);
    } else {
      queue.push(value);
    }
  }

  async queueItems(key: string): Promise<Buffer[]> {
    return [...(this.queues.get(key) || [])];
  }

  override async queueGather(key: string): Promise<Buffer> {
    const queue = this.queues.get(key);
    if (queue?.length) return queue.shift();
  }

  override async queueGatherBlocking(key: string): Promise<Buffer> {
    const itemInQueue = await this.queueGather(key);
    if (itemInQueue) return itemInQueue;
    if (!this.blockingGathers.has(key)) {
      this.blockingGathers.set(key, []);
    }
    return new Promise((resolve) => {
      this.blockingGathers.get(key).push(resolve);
    });
  }

  override async queueClear(key: string): Promise<void> {
    this.queues.delete(key);
  }
}
